<?php

namespace App\Services;

use App\Jobs\UpsertBookingFromVeVs;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use RuntimeException;

class VeVsImporter
{
    public function __construct(
        protected VeVsApi $api,
    ) {
    }

    /**
     * Import / refresh a single booking by its VEVS reference (ref_id).
     *
     * - Calls the VEVS API
     * - Unwraps the "data" key
     * - Validates minimum required fields
     * - Runs UpsertBookingFromVeVs synchronously
     *
     * @return array<string,mixed> The normalized row used for upsert.
     */
    public function importByRef(string $reference): array
    {
        $payload = $this->api->reservationByReference($reference);

        if (empty($payload)) {
            throw new RuntimeException("VEVS: Reservation {$reference} not found or invalid.");
        }

        // Unwrap the "data" key if present (current API behaviour)
        $rawRow = $payload['data'] ?? $payload;

        if (! is_array($rawRow)) {
            throw new RuntimeException("VEVS: Unexpected reservation payload format for {$reference}.");
        }

        // If you ever want to massage / map keys, do it here.
        $row = $this->normalizeReservationRow($rawRow);

        $this->validateRow($row, $reference);

        // Perform the actual upsert into our Bookings table
        (new UpsertBookingFromVeVs($row))->handle();

        return $row;
    }

    /**
     * Minimal validation to ensure the row has what our Upsert job expects.
     */
    protected function validateRow(array $row, string $reference): void
    {
        $validator = Validator::make($row, [
            // DO NOT require "id" – we can fall back to uuid/booking_id
            'ref_id' => ['required'],
            'car_id' => ['required'],
            'from'   => ['required'],
            'to'     => ['required'],
        ]);

        if ($validator->fails()) {
            $msg = 'Validation failed: ' . $validator->errors()->first();

            Log::warning('[VeVsImporter] ' . $msg, [
                'reference' => $reference,
                'row_keys'  => array_keys($row),
            ]);

            throw new RuntimeException($msg);
        }
    }

    /**
     * Hook for any future mapping / normalisation.
     *
     * For now, the VEVS row already has the shape UpsertBookingFromVeVs expects,
     * so we just return it as-is.
     *
     * @param  array<string,mixed>  $row
     * @return array<string,mixed>
     */
    protected function normalizeReservationRow(array $row): array
    {
        // Example of where we could map alternative keys:
        // $row['ref_id'] ??= $row['ReferenceID'] ?? $row['reference'] ?? null;
        // $row['car_id'] ??= $row['vehicle_id'] ?? null;

        return $row;
    }
}
