<?php

declare(strict_types=1);

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class VevsClient
{
    public function __construct(
        protected string $baseUrl, // e.g. https://www.dreamdrives.co.nz/api/.../Reservation
        protected string $apiKey = '', // not used for Dream Drives, but kept for Jimny compatibility
    ) {
    }

    /**
     * Fetch a single reservation by its reference from VEVS.
     *
     * For Dream Drives, this will hit:
     *   https://www.dreamdrives.co.nz/api/.../Reservation?format=json&ref_id={REF}
     */
    public function fetchReservationByRef(string $ref): ?array
    {
        $response = Http::get($this->baseUrl, [
            'format' => 'json',
            'ref_id' => $ref,
            // If Jimny needs an API key as a query param, you can add it here conditionally,
            // e.g. 'api_key' => $this->apiKey when not empty.
        ]);

        Log::info('VEVS fetchReservationByRef response', [
            'base_url' => $this->baseUrl,
            'ref'      => $ref,
            'status'   => $response->status(),
            'body'     => $response->body(),
        ]);

        if (! $response->successful()) {
            return null;
        }

        // For this endpoint, the body *should* be JSON already.
        $data = $response->json();

        if (! is_array($data)) {
            return null;
        }

        return $data;
    }
}
