<?php

namespace App\Support\Payments;

use App\Mail\PaymentRequestMail;
use App\Models\Attachment;
use App\Models\Job;
use App\Models\Payment;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Str;

class OneOffPaymentService
{
    public function createOneOffPayment(Job $job, array $payload, ?UploadedFile $attachment = null): array
    {
        $amount = (int) ($payload['amount_cents'] ?? 0);
        abort_if($amount < 100, 422, 'Amount must be at least $1.00');

        return DB::transaction(function () use ($job, $payload, $attachment, $amount) {
            $payment = Payment::create([
                'job_id'       => $job->id,
                'amount_cents' => $amount,
                'currency'     => $payload['currency'] ?? 'NZD',
                'status'       => 'pending',
                'type'         => 'one_off',
                'charge_mode'  => $payload['mode'] ?? 'request_link',
                'reason'       => $payload['reason'] ?? null,
                'public_token' => (string) Str::uuid(),
                'gateway'      => 'stripe',
            ]);

            if ($attachment instanceof UploadedFile) {
                $path = $attachment->store("payments/{$payment->id}", ['disk' => config('filesystems.default')]);
                $payment->attachments()->create([
                    'disk'          => config('filesystems.default'),
                    'path'          => $path,
                    'original_name' => $attachment->getClientOriginalName(),
                    'mime'          => $attachment->getClientMimeType(),
                    'size_bytes'    => $attachment->getSize(),
                ]);
            }

            return match ($payment->charge_mode) {
                'off_session'  => $this->attemptOffSessionCharge($payment),
                'request_link' => $this->initCheckoutAndEmail($payment),
                default        => ['status' => 'error', 'message' => 'Unknown charge mode.'],
            };
        });
    }

    protected function stripe(): \Stripe\StripeClient
    {
        return new \Stripe\StripeClient((string) config('services.stripe.secret'));
    }

    protected function attemptOffSessionCharge(Payment $p): array
    {
        $job = $p->job;
        $customerStripeId = $job?->customer?->stripe_customer_id;

        if (!$customerStripeId) {
            // No card on file → fall back to link
            $p->update(['charge_mode' => 'request_link']);
            return $this->initCheckoutAndEmail($p, 'No saved card; sent payment link instead.');
        }

        $meta = [
            'kind'       => 'one_off',
            'payment_id' => $p->id,
            'job_id'     => $job->id,
            'job_ref'    => $job->reference ?? null,
        ];

        try {
            $pi = $this->stripe()->paymentIntents->create([
                'amount'       => $p->amount_cents,
                'currency'     => strtolower($p->currency ?? 'nzd'),
                'customer'     => $customerStripeId,
                'off_session'  => true,
                'confirm'      => true,
                'metadata'     => $meta,
            ]);

            $p->update(['status' => 'succeeded', 'gateway_meta' => array_merge((array) $p->gateway_meta, ['pi' => $pi->id])]);
            // Fire your successful event, send receipt, etc.
            return ['status' => 'succeeded'];
        } catch (\Stripe\Exception\CardException $e) {
            // 3DS or decline → flip to link & notify
            $p->update([
                'charge_mode'  => 'request_link',
                'gateway_meta' => array_merge((array) $p->gateway_meta, ['error' => $e->getMessage()]),
            ]);
            return $this->initCheckoutAndEmail($p, 'Card requires authentication; sent payment link instead.');
        }
    }

    protected function initCheckoutAndEmail(Payment $p, ?string $note = null): array
    {
        $session = $this->stripe()->checkout->sessions->create([
            'mode' => 'payment',
            'payment_intent_data' => [
                'metadata' => [
                    'payment_id' => $p->id,
                    'job_id'     => $p->job_id,
                    'kind'       => 'one_off',
                ],
            ],
            'line_items' => [[
                'price_data' => [
                    'currency'     => strtolower($p->currency ?? 'nzd'),
                    'product_data' => [
                        'name'        => 'One-off charge',
                        'description' => mb_substr($p->reason ?? 'Additional charge', 0, 500),
                    ],
                    'unit_amount'  => $p->amount_cents,
                ],
                'quantity' => 1,
            ]],
            'success_url' => route('payments.public.thanks', $p->public_token),
            'cancel_url'  => route('payments.public.cancel', $p->public_token),
            'metadata'    => ['payment_id' => $p->id, 'job_id' => $p->job_id, 'kind' => 'one_off'],
        ]);

        $p->update(['gateway_meta' => array_merge((array) $p->gateway_meta, ['checkout_session' => $session->id])]);

        // Email payment request + show attachment list (links)
        $to = optional($p->job->customer)->email;
        if ($to) {
            Mail::to($to)->send(new PaymentRequestMail($p, $session->url, $note));
        }

        return ['status' => 'pending', 'checkout_url' => $session->url, 'message' => $note];
    }
}
