<?php
// database/migrations/2025_09_05_192734_make_booking_id_nullable_on_payments.php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        if (! Schema::hasTable('payments') || ! Schema::hasColumn('payments', 'booking_id')) {
            return;
        }

        $this->dropForeignKeyIfExists('payments', 'booking_id');

        // Requires doctrine/dbal for ->change()
        Schema::table('payments', function (Blueprint $table) {
            $table->unsignedBigInteger('booking_id')->nullable()->change();
        });

        // Recreate FK with ON DELETE SET NULL (safe if booking is removed)
        Schema::table('payments', function (Blueprint $table) {
            // Name it predictably so future drops are easy
            $table->foreign('booking_id', 'payments_booking_id_foreign')
                ->references('id')
                ->on('bookings')
                ->nullOnDelete();
        });
    }

    public function down(): void
    {
        if (! Schema::hasTable('payments') || ! Schema::hasColumn('payments', 'booking_id')) {
            return;
        }

        $this->dropForeignKeyIfExists('payments', 'booking_id');

        // Warning: this will fail if rows exist with NULL booking_id.
        Schema::table('payments', function (Blueprint $table) {
            $table->unsignedBigInteger('booking_id')->nullable(false)->change();
        });

        Schema::table('payments', function (Blueprint $table) {
            $table->foreign('booking_id', 'payments_booking_id_foreign')
                ->references('id')
                ->on('bookings')
                ->cascadeOnDelete();
        });
    }

    /**
     * Find and drop any FK on $table.$column (works across environments where names differ).
     */
    protected function dropForeignKeyIfExists(string $table, string $column): void
    {
        // Look up any FK constraint on this table/column
        $constraints = DB::select(<<<SQL
            SELECT CONSTRAINT_NAME
            FROM information_schema.KEY_COLUMN_USAGE
            WHERE TABLE_SCHEMA = DATABASE()
              AND TABLE_NAME = ?
              AND COLUMN_NAME = ?
              AND REFERENCED_TABLE_NAME IS NOT NULL
        SQL, [$table, $column]);

        foreach ($constraints as $row) {
            $name = $row->CONSTRAINT_NAME ?? null;
            if (! $name) continue;

            // Drop by exact name (wrap in try to be idempotent)
            try {
                DB::statement(sprintf(
                    'ALTER TABLE `%s` DROP FOREIGN KEY `%s`',
                    str_replace('`', '``', $table),
                    str_replace('`', '``', $name)
                ));
            } catch (\Throwable $e) {
                // no-op if already gone
            }
        }
    }
};
