<?php
// database/migrations/2025_09_12_014519_add_stripe_columns_to_payments_table.php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        if (! Schema::hasTable('payments')) {
            return;
        }

        // 1) Add Stripe columns if missing
        Schema::table('payments', function (Blueprint $table) {
            if (! Schema::hasColumn('payments', 'stripe_payment_intent')) {
                $table->string('stripe_payment_intent')->nullable()->after('id');
            }
            if (! Schema::hasColumn('payments', 'stripe_charge_id')) {
                $table->string('stripe_charge_id')->nullable()->after('stripe_payment_intent');
            }
        });

        // 2) Ensure status/type are short enough for composite index (utf8mb4 safe)
        //    Requires doctrine/dbal for ->change()
        Schema::table('payments', function (Blueprint $table) {
            if (Schema::hasColumn('payments', 'status')) {
                $table->string('status', 50)->change();
            }
            if (Schema::hasColumn('payments', 'type')) {
                $table->string('type', 50)->change();
            }
        });

        // 3) Drop existing composite index if present (no fatal if it doesn't exist)
        try {
            Schema::table('payments', function (Blueprint $table) {
                $table->dropIndex('payments_job_id_status_type_index');
            });
        } catch (\Throwable $e) {
            // index didn't exist; ignore
        }

        // 4) Add composite index (job_id + status + type)
        try {
            Schema::table('payments', function (Blueprint $table) {
                $table->index(['job_id', 'status', 'type'], 'payments_job_id_status_type_index');
            });
        } catch (\Throwable $e) {
            // If this fails again due to lengths, double-check step (2) ran and that doctrine/dbal is installed
        }
    }

    public function down(): void
    {
        if (! Schema::hasTable('payments')) {
            return;
        }

        // Drop Stripe columns (best-effort)
        Schema::table('payments', function (Blueprint $table) {
            if (Schema::hasColumn('payments', 'stripe_charge_id')) {
                $table->dropColumn('stripe_charge_id');
            }
            if (Schema::hasColumn('payments', 'stripe_payment_intent')) {
                $table->dropColumn('stripe_payment_intent');
            }
        });

        // Drop the composite index if present
        try {
            Schema::table('payments', function (Blueprint $table) {
                $table->dropIndex('payments_job_id_status_type_index');
            });
        } catch (\Throwable $e) {
            // ignore
        }
    }
};
