<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /** Check if a specific index name exists on a table. */
    private function indexExists(string $table, string $index): bool
    {
        $rows = DB::select("SHOW INDEX FROM `$table` WHERE Key_name = ?", [$index]);
        return !empty($rows);
    }

    /** Check if a column exists (wrapper for readability). */
    private function col(string $table, string $col): bool
    {
        return Schema::hasColumn($table, $col);
    }

    public function up(): void
    {
        // 1) Ensure paid_at exists (nullable) and is indexed for querying
        if (!$this->col('payments', 'paid_at')) {
            Schema::table('payments', function (Blueprint $table) {
                $table->dateTime('paid_at')->nullable()->after('details');
            });
        }

        // Add an index on paid_at if it’s not already present
        if ($this->col('payments', 'paid_at') && !$this->indexExists('payments', 'payments_paid_at_index')) {
            Schema::table('payments', function (Blueprint $table) {
                $table->index('paid_at', 'payments_paid_at_index');
            });
        }

        // 2) Only add booking_id index if the column exists AND the index does not yet exist
        if ($this->col('payments', 'booking_id') && !$this->indexExists('payments', 'payments_booking_id_index')) {
            Schema::table('payments', function (Blueprint $table) {
                $table->index('booking_id', 'payments_booking_id_index');
            });
        }

        // (Optional) If you previously added a duplicate under a different name, drop it here
        // Example:
        // if ($this->indexExists('payments', 'booking_id')) {
        //     Schema::table('payments', fn (Blueprint $t) => $t->dropIndex('booking_id'));
        // }
    }

    public function down(): void
    {
        // Be conservative: only drop what we added
        if ($this->indexExists('payments', 'payments_paid_at_index')) {
            Schema::table('payments', fn (Blueprint $t) => $t->dropIndex('payments_paid_at_index'));
        }

        // Do NOT drop the column by default in down() if data may exist.
        // If you want full reversibility, uncomment:
        // if (Schema::hasColumn('payments', 'paid_at')) {
        //     Schema::table('payments', fn (Blueprint $t) => $t->dropColumn('paid_at'));
        // }

        if ($this->indexExists('payments', 'payments_booking_id_index')) {
            Schema::table('payments', fn (Blueprint $t) => $t->dropIndex('payments_booking_id_index'));
        }
    }
};
