<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;

return new class extends Migration {
    public function up(): void
    {
        // 1) Ensure required columns exist (guarded)
        Schema::table('payments', function (Blueprint $t) {
            if (!Schema::hasColumn('payments', 'paid_at')) {
            $table->timestamp('paid_at')->nullable()->after('details');
            }
            if (!Schema::hasColumn('payments', 'amount_cents')) {
                $t->integer('amount_cents')->default(0)->index();
            }
            if (!Schema::hasColumn('payments', 'currency')) {
                $t->string('currency', 16)->default('NZD')->index();
            }
            if (!Schema::hasColumn('payments', 'status')) {
                $t->string('status', 32)->default('pending')->index();
            }
            if (!Schema::hasColumn('payments', 'type')) {
                $t->string('type', 32)->default('charge')->index();
            }

            if (!Schema::hasColumn('payments', 'provider')) {
                $t->string('provider', 32)->nullable()->index();
            }
            if (!Schema::hasColumn('payments', 'stripe_payment_intent')) {
                $t->string('stripe_payment_intent', 64)->nullable()->index();
            }
            if (!Schema::hasColumn('payments', 'stripe_payment_method')) {
                $t->string('stripe_payment_method', 64)->nullable()->index();
            }
            if (!Schema::hasColumn('payments', 'stripe_charge')) {
                $t->string('stripe_charge', 64)->nullable()->index();
            }
            if (!Schema::hasColumn('payments', 'meta')) {
                $t->json('meta')->nullable();
            }
        });

        // 2) Backfill from legacy columns if new ones are empty
        DB::table('payments')
            ->select([
                'id',
                'stripe_payment_intent',
                'stripe_payment_method',
                'stripe_charge',
                'meta',
                'stripe_payment_intent_id',
                'stripe_payment_method_id',
                'stripe_charge_id',
                'metadata',
                'details',
            ])
            ->orderBy('id')
            ->chunk(1000, function ($rows) {
                foreach ($rows as $row) {
                    $updates = [];

                    if (empty($row->stripe_payment_intent) && !empty($row->stripe_payment_intent_id)) {
                        $updates['stripe_payment_intent'] = $row->stripe_payment_intent_id;
                    }
                    if (empty($row->stripe_payment_method) && !empty($row->stripe_payment_method_id)) {
                        $updates['stripe_payment_method'] = $row->stripe_payment_method_id;
                    }
                    if (empty($row->stripe_charge) && !empty($row->stripe_charge_id)) {
                        $updates['stripe_charge'] = $row->stripe_charge_id;
                    }

                    if (is_null($row->meta)) {
                        $candidate = null;
                        if (!empty($row->metadata)) {
                            $candidate = $row->metadata;
                        } elseif (!empty($row->details)) {
                            $candidate = $row->details;
                        }

                        if (!is_null($candidate)) {
                            $decoded = json_decode($candidate, true);
                            if (json_last_error() === JSON_ERROR_NONE) {
                                $updates['meta'] = json_encode($decoded);
                            } else {
                                $updates['meta'] = json_encode(['notes' => (string) $candidate]);
                            }
                        }
                    }

                    if ($updates) {
                        DB::table('payments')->where('id', $row->id)->update($updates);
                    }
                }
            });

        // Note: We intentionally do NOT add any new indexes here to avoid duplicate index errors.
        // If you want extra indexes later, do it in a separate guarded migration.
    }

    public function down(): void
    {
        Schema::table('payments', function (Blueprint $t) {
            foreach ([
                'provider',
                'stripe_payment_intent',
                'stripe_payment_method',
                'stripe_charge',
                'meta',
            ] as $col) {
                if (Schema::hasColumn('payments', $col)) {
                    $t->dropColumn($col);
                }
            }
        });
    }
};
