<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        if (! Schema::hasTable('bookings')) {
            return;
        }

        Schema::table('bookings', function (Blueprint $t) {
            // Core refs
            if (! Schema::hasColumn('bookings', 'external_reference')) {
                $t->string('external_reference', 191)->nullable()->index()->after('id');
            }

            if (! Schema::hasColumn('bookings', 'reference')) {
                $t->string('reference', 191)->nullable()->after('external_reference');
                $t->index('reference');
            }

            // Customer relation
            if (! Schema::hasColumn('bookings', 'customer_id')) {
                $t->foreignId('customer_id')->nullable()->constrained()->nullOnDelete()->after('reference');
            }

            // Dates (what your import code currently uses)
            if (! Schema::hasColumn('bookings', 'start_date')) {
                $t->dateTime('start_date')->nullable()->after('customer_id');
            }
            if (! Schema::hasColumn('bookings', 'end_date')) {
                $t->dateTime('end_date')->nullable()->after('start_date');
            }

            // Basic booking details
            if (! Schema::hasColumn('bookings', 'vehicle_name')) {
                $t->string('vehicle_name', 191)->nullable()->after('end_date');
            }
            if (! Schema::hasColumn('bookings', 'total_amount')) {
                // store in cents
                $t->integer('total_amount')->nullable()->after('vehicle_name');
            }
            if (! Schema::hasColumn('bookings', 'currency')) {
                $t->string('currency', 3)->nullable()->after('total_amount');
            }

            // Customer portal token
            if (! Schema::hasColumn('bookings', 'portal_token')) {
                $t->string('portal_token', 191)->nullable()->unique()->after('currency');
            }
        });
    }

    public function down(): void
    {
        if (! Schema::hasTable('bookings')) {
            return;
        }

        Schema::table('bookings', function (Blueprint $t) {
            // Drop columns only if they exist. Keep order to satisfy FKs / indexes.
            if (Schema::hasColumn('bookings', 'portal_token')) {
                // Laravel will infer the unique index name when created by column
                $t->dropUnique('bookings_portal_token_unique');
                $t->dropColumn('portal_token');
            }
            if (Schema::hasColumn('bookings', 'currency')) {
                $t->dropColumn('currency');
            }
            if (Schema::hasColumn('bookings', 'total_amount')) {
                $t->dropColumn('total_amount');
            }
            if (Schema::hasColumn('bookings', 'vehicle_name')) {
                $t->dropColumn('vehicle_name');
            }
            if (Schema::hasColumn('bookings', 'end_date')) {
                $t->dropColumn('end_date');
            }
            if (Schema::hasColumn('bookings', 'start_date')) {
                $t->dropColumn('start_date');
            }
            if (Schema::hasColumn('bookings', 'customer_id')) {
                $t->dropConstrainedForeignId('customer_id');
            }
            if (Schema::hasColumn('bookings', 'reference')) {
                $t->dropIndex('bookings_reference_index');
                $t->dropColumn('reference');
            }
            if (Schema::hasColumn('bookings', 'external_reference')) {
                $t->dropIndex('bookings_external_reference_index');
                $t->dropColumn('external_reference');
            }
        });
    }
};
