<?php
// database/migrations/2025_09_19_114944_create_payment_links_table.php
// Evolution/guarded migration: never re-create table or duplicate indexes.

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\DB;

return new class extends Migration {
    public function up(): void
    {
        // If table exists, just patch missing bits and avoid duplicate indexes.
        if (Schema::hasTable('payment_links')) {
            Schema::table('payment_links', function (Blueprint $table) {
                if (! Schema::hasColumn('payment_links', 'tenant_id')) {
                    $table->unsignedBigInteger('tenant_id')->nullable()->after('payment_id');
                    $table->index('tenant_id');
                }
                if (! Schema::hasColumn('payment_links', 'public_token')) {
                    $table->string('public_token', 64)->after('tenant_id');
                }
                if (! Schema::hasColumn('payment_links', 'expires_at')) {
                    $table->timestamp('expires_at')->nullable()->after('public_token');
                }
                if (! Schema::hasColumn('payment_links', 'clicked_at')) {
                    $table->timestamp('clicked_at')->nullable()->after('expires_at');
                }
                if (! Schema::hasColumn('payment_links', 'meta')) {
                    $table->json('meta')->nullable()->after('clicked_at');
                }
                if (! Schema::hasColumn('payment_links', 'created_at')) {
                    $table->timestamps();
                }
            });

            // Ensure UNIQUE(public_token) exists exactly once
            if (Schema::hasColumn('payment_links', 'public_token')) {
                $dbName = DB::getDatabaseName();
                $hasUnique = DB::table('information_schema.STATISTICS')
                    ->where('TABLE_SCHEMA', $dbName)
                    ->where('TABLE_NAME', 'payment_links')
                    ->where('COLUMN_NAME', 'public_token')
                    ->where('NON_UNIQUE', 0)
                    ->exists();

                if (! $hasUnique) {
                    // name aligns with Laravel's default naming
                    DB::statement('ALTER TABLE `payment_links` ADD UNIQUE `payment_links_public_token_unique` (`public_token`)');
                }
            }

            return; // done
        }

        // Fresh create on new databases only
        Schema::create('payment_links', function (Blueprint $table) {
            $table->id();
            $table->unsignedBigInteger('payment_id');
            $table->unsignedBigInteger('tenant_id')->nullable();
            $table->string('public_token', 64);
            $table->timestamp('expires_at')->nullable();
            $table->timestamp('clicked_at')->nullable();
            $table->json('meta')->nullable();
            $table->timestamps();

            $table->index('payment_id');
            $table->index('tenant_id');
            $table->unique('public_token');
        });
    }

    public function down(): void
    {
        // no destructive rollback
    }
};
