<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;

return new class extends Migration {
    public function up(): void
    {
        // 1) Add columns if missing
        Schema::table('payments', function (Blueprint $t) {
            if (!Schema::hasColumn('payments', 'psp')) {
                $t->string('psp')->nullable()->after('status'); // e.g. 'stripe'
            }
            if (!Schema::hasColumn('payments', 'stripe_payment_intent_id')) {
                $t->string('stripe_payment_intent_id')->nullable()->after('psp');
            }
        });

        // 2) (Safety) Remove duplicate rows per (psp, stripe_payment_intent_id), keeping the newest.
        //    This prevents the unique index from failing.
        DB::statement("
            DELETE p1 FROM payments p1
            INNER JOIN payments p2
                ON p1.psp = p2.psp
               AND p1.stripe_payment_intent_id = p2.stripe_payment_intent_id
               AND p1.id < p2.id
            WHERE p1.psp IS NOT NULL
              AND p1.stripe_payment_intent_id IS NOT NULL
        ");

        // 3) Add the unique index if it doesn't exist (using information_schema; no Doctrine).
        $database = DB::getDatabaseName();
        $exists = DB::table('information_schema.statistics')
            ->where('table_schema', $database)
            ->where('table_name', 'payments')
            ->where('index_name', 'payments_psp_pi_unique')
            ->exists();

        if (! $exists) {
            Schema::table('payments', function (Blueprint $t) {
                $t->unique(['psp', 'stripe_payment_intent_id'], 'payments_psp_pi_unique');
            });
        }
    }

    public function down(): void
    {
        // Drop unique if present
        $database = DB::getDatabaseName();
        $exists = DB::table('information_schema.statistics')
            ->where('table_schema', $database)
            ->where('table_name', 'payments')
            ->where('index_name', 'payments_psp_pi_unique')
            ->exists();

        if ($exists) {
            Schema::table('payments', function (Blueprint $t) {
                $t->dropUnique('payments_psp_pi_unique');
            });
        }

        Schema::table('payments', function (Blueprint $t) {
            if (Schema::hasColumn('payments', 'stripe_payment_intent_id')) {
                $t->dropColumn('stripe_payment_intent_id');
            }
            if (Schema::hasColumn('payments', 'psp')) {
                $t->dropColumn('psp');
            }
        });
    }
};
