<?php

declare(strict_types=1);

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        Schema::table('jobs', function (Blueprint $table): void {
            // money buckets (all cents)
            $moneyCols = [
                'base_rental_cents',
                'extra_mileage_fee_cents',
                'extras_cents',
                'other_fees_cents',
                'insurance_cents',
                'vat_cents',
                'total_price_cents',
            ];

            foreach ($moneyCols as $col) {
                if (! Schema::hasColumn('jobs', $col)) {
                    $table->bigInteger($col)->nullable()->after('charge_amount_cents');
                }
            }

            if (! Schema::hasColumn('jobs', 'currency_code')) {
                $table->string('currency_code', 3)->nullable()->after('total_price_cents');
            }

            // owner share overrides
            if (! Schema::hasColumn('jobs', 'owner_share_mode')) {
                $table->string('owner_share_mode')
                    ->nullable() // null = global model
                    ->after('currency_code');
            }

            if (! Schema::hasColumn('jobs', 'owner_share_percent_override')) {
                $table->unsignedTinyInteger('owner_share_percent_override')
                    ->nullable()
                    ->after('owner_share_mode');
            }

            if (! Schema::hasColumn('jobs', 'owner_fixed_amount_cents')) {
                $table->bigInteger('owner_fixed_amount_cents')
                    ->nullable()
                    ->after('owner_share_percent_override');
            }

            // flag so we don’t double-generate ledger entries
            if (! Schema::hasColumn('jobs', 'owner_earnings_generated_at')) {
                $table->timestamp('owner_earnings_generated_at')
                    ->nullable()
                    ->after('owner_fixed_amount_cents');
            }
        });
    }

    public function down(): void
    {
        Schema::table('jobs', function (Blueprint $table): void {
            $moneyCols = [
                'base_rental_cents',
                'extra_mileage_fee_cents',
                'extras_cents',
                'other_fees_cents',
                'insurance_cents',
                'vat_cents',
                'total_price_cents',
            ];

            foreach ($moneyCols as $col) {
                if (Schema::hasColumn('jobs', $col)) {
                    $table->dropColumn($col);
                }
            }

            foreach ([
                'currency_code',
                'owner_share_mode',
                'owner_share_percent_override',
                'owner_fixed_amount_cents',
                'owner_earnings_generated_at',
            ] as $col) {
                if (Schema::hasColumn('jobs', $col)) {
                    $table->dropColumn($col);
                }
            }
        });
    }
};
