@php
/**
 * One‑off Payment Page (similar to existing pay page)
 * File: resources/views/pay/oneoff.blade.php
 *
 * Expects from controller:
 *  - $payment: App\Models\Payment (id, amount_cents, currency, description, status, reference)
 *  - $amountCents: int  (final charge amount in cents)  // allows overrides/fees
 *  - $currency: string  (e.g., 'NZD')
 *  - $customerEmail: ?string
 *  - $clientSecret: ?string  // for Stripe PaymentIntent (inline card via Elements)
 *  - $checkoutUrl:  ?string  // for Stripe Checkout Session
 *  - $stripePk: string       // Stripe publishable key
 *  - $attachments: Illuminate\Support\Collection<Attachment>
 *  - $brand: array{name:string,logo?:string,support_email?:string,privacy_url?:string,terms_url?:string}
 *  - $allowAttachmentUpload: bool
 *  - $showBankDetails: bool
 *  - $bankDetails: array{account_name?:string,account_number?:string,reference?:string,instructions?:string}
 */
@endphp

@extends('layouts.public')

@section('title', 'Secure Payment')

@section('head')
  <meta name="robots" content="noindex,nofollow">
  <script src="https://js.stripe.com/v3/"></script>
  <style>
    .PayCard { max-width: 820px; margin: 2rem auto; }
    .card { border: 1px solid #e5e7eb; border-radius: 16px; box-shadow: 0 8px 24px rgba(0,0,0,0.06); overflow: hidden; }
    .card-header { padding: 20px 24px; background: #0f172a; color: #fff; display: flex; align-items: center; gap: 12px; }
    .card-body { padding: 24px; background: #fff; }
    .grid { display: grid; grid-template-columns: 1.2fr 1fr; gap: 24px; }
    .badge { display:inline-flex; align-items:center; gap:8px; padding:6px 10px; border-radius:999px; background:#f1f5f9; font-size:12px; }
    .muted { color:#64748b; }
    .money { font-weight: 800; font-size: 28px; letter-spacing: .2px; }
    .btn { display:inline-flex; align-items:center; justify-content:center; gap:8px; padding:12px 16px; border-radius:12px; border:1px solid #0f172a; background:#0f172a; color:#fff; font-weight:600; cursor:pointer; text-decoration:none; }
    .btn.secondary { background:#fff; color:#0f172a; }
    .btn:disabled { opacity:.6; cursor:not-allowed; }
    .row { display:flex; gap:12px; align-items:center; }
    .field { display:flex; flex-direction:column; gap:6px; margin-bottom:14px; }
    .field input[type="text"], .field input[type="email"], .field input[type="file"] { border:1px solid #cbd5e1; border-radius:10px; padding:12px 14px; font-size:14px; }
    .callout { border:1px dashed #cbd5e1; border-radius:12px; padding:14px; background:#f8fafc; }
    .pilltabs { display:flex; background:#f1f5f9; padding:6px; border-radius:999px; gap:6px; }
    .pilltabs button { border:none; background:transparent; padding:8px 12px; border-radius:999px; cursor:pointer; font-weight:600; }
    .pilltabs button[aria-pressed="true"]{ background:#fff; box-shadow:0 2px 6px rgba(0,0,0,.06); }
    .stripe-row { border:1px solid #e2e8f0; border-radius:12px; padding:12px; }
    .logo { height:28px; width:auto; }
    @media (max-width: 920px){ .grid{ grid-template-columns: 1fr; } }
  </style>
@endsection

@section('content')
<div class="PayCard">
  <div class="card">
    <div class="card-header">
      @if (!empty($brand['logo']))
        <img class="logo" src="{{ $brand['logo'] }}" alt="{{ $brand['name'] ?? 'Brand' }} logo">
      @endif
      <div>
        <div style="font-size:14px; opacity:.8">Secure payment</div>
        <div style="font-size:18px; font-weight:700">{{ $brand['name'] ?? config('app.name') }}</div>
      </div>
      <div style="margin-left:auto">
        <span class="badge"><svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M12 2L3 7v10l9 5 9-5V7l-9-5Z" stroke="#334155" stroke-width="1.5"/><path d="M7.5 12 11 15.5l5.5-7" stroke="#16a34a" stroke-width="1.8"/></svg> PCI compliant</span>
      </div>
    </div>

    <div class="card-body">
      <div class="grid">
        <!-- Left: Payment form -->
        <div>
          <h2 style="margin:0 0 6px 0; font-size:22px; font-weight:800;">Pay now</h2>
          <div class="muted">Reference: <strong>{{ $payment->reference ?? ('PMT-'.$payment->id) }}</strong></div>

          <div class="row" style="margin:14px 0 10px; align-items:baseline; gap:10px;">
            <div class="money">{{ strtoupper($currency) }} {{ number_format($amountCents/100, 2) }}</div>
            @if ($payment->description)
              <div class="muted">· {{ $payment->description }}</div>
            @endif
          </div>

          <!-- Method toggle: Elements vs Checkout (shown only if both available) -->
          @php
            $hasElements = filled($clientSecret);
            $hasCheckout = filled($checkoutUrl);
          @endphp
          @if ($hasElements && $hasCheckout)
            <div class="pilltabs" role="tablist" aria-label="Choose payment method" style="margin-bottom:14px;">
              <button type="button" class="js-tab" data-target="#tab-elements" aria-pressed="true">Card on this page</button>
              <button type="button" class="js-tab" data-target="#tab-checkout" aria-pressed="false">Stripe Checkout</button>
            </div>
          @endif

          <!-- Inline Elements form -->
          @if ($hasElements)
          <div id="tab-elements" class="tabpanel" role="region" aria-label="Card payment">
            <form id="elements-form" action="{{ route('payments.pay.inline', $payment) }}" method="post">
              @csrf
              <div class="field">
                <label for="payer_email">Email (for receipt)</label>
                <input id="payer_email" name="email" type="email" value="{{ old('email', $customerEmail) }}" autocomplete="email" required>
              </div>

              <div class="stripe-row">
                <div id="link-authentication-element"></div>
                <div id="payment-element" style="margin-top:10px;"></div>
              </div>

              <div class="field" style="margin-top:10px;">
                <label class="row"><input type="checkbox" name="save_card" value="1"> <span class="muted">Save card for future charges (off‑session)</span></label>
              </div>

              <button id="elements-submit" class="btn" type="submit" data-loading-text="Processing…">Pay {{ strtoupper($currency) }} {{ number_format($amountCents/100,2) }}</button>
              <div id="elements-error" class="muted" style="margin-top:10px;color:#b91c1c;" role="alert"></div>
            </form>
          </div>
          @endif

          <!-- Checkout redirect option -->
          @if ($hasCheckout)
          <div id="tab-checkout" class="tabpanel" role="region" aria-label="Stripe Checkout" @if($hasElements) style="display:none" @endif>
            <div class="callout" style="margin-bottom:10px;">You'll be redirected to Stripe’s secure Checkout page to complete your payment.</div>
            <a class="btn" href="{{ $checkoutUrl }}">Continue to Stripe Checkout</a>
          </div>
          @endif

          <!-- Optional bank transfer instructions -->
          @if ($showBankDetails ?? false)
            <details style="margin-top:16px;">
              <summary class="row" style="cursor:pointer; font-weight:700;">Prefer bank transfer?</summary>
              <div class="callout" style="margin-top:10px;">
                <div><strong>Account name:</strong> {{ $bankDetails['account_name'] ?? '' }}</div>
                <div><strong>Account number:</strong> {{ $bankDetails['account_number'] ?? '' }}</div>
                <div><strong>Reference:</strong> {{ $bankDetails['reference'] ?? ($payment->reference ?? $payment->id) }}</div>
                @if (!empty($bankDetails['instructions']))
                  <p class="muted" style="margin:8px 0 0;">{{ $bankDetails['instructions'] }}</p>
                @endif
              </div>
            </details>
          @endif
        </div>

        <!-- Right: Summary & attachments -->
        <aside>
          <div class="callout">
            <div class="row" style="justify-content:space-between;">
              <div>
                <div class="muted">Amount due</div>
                <div class="money">{{ strtoupper($currency) }} {{ number_format($amountCents/100,2) }}</div>
              </div>
              <div class="muted" style="text-align:right;">Payment #{{ $payment->id }}</div>
            </div>
            @if($payment->description)
              <div class="muted" style="margin-top:8px;">{{ $payment->description }}</div>
            @endif
          </div>

          <div style="margin-top:16px;">
            <h3 style="margin:0 0 8px 0; font-size:16px; font-weight:800;">Attachments</h3>
            @if ($attachments && $attachments->count())
              <ul style="margin:0; padding-left:18px;">
                @foreach ($attachments as $file)
                  <li><a href="{{ route('attachments.show', $file) }}" target="_blank" rel="noopener">{{ $file->original_name }}</a></li>
                @endforeach
              </ul>
            @else
              <div class="muted">No files attached.</div>
            @endif

            @if ($allowAttachmentUpload ?? false)
              <form action="{{ route('payments.attachments.store', $payment) }}" method="post" enctype="multipart/form-data" style="margin-top:10px;">
                @csrf
                <div class="field">
                  <label for="evidence">Add evidence (PDF/JPG/PNG)</label>
                  <input id="evidence" name="files[]" type="file" accept="application/pdf,image/*" multiple>
                </div>
                <button class="btn secondary" type="submit">Upload</button>
              </form>
            @endif
          </div>

          <div class="muted" style="margin-top:16px; font-size:12px;">
            By paying you agree to our <a href="{{ $brand['terms_url'] ?? '#' }}" target="_blank" rel="noopener">Terms</a> and <a href="{{ $brand['privacy_url'] ?? '#' }}" target="_blank" rel="noopener">Privacy Policy</a>.
            Need help? <a href="mailto:{{ $brand['support_email'] ?? 'hello@example.com' }}">{{ $brand['support_email'] ?? 'hello@example.com' }}</a>
          </div>
        </aside>
      </div>
    </div>
  </div>
</div>

@if ($hasElements)
<script>
  (function(){
    const stripe = Stripe(@json($stripePk));
    const clientSecret = @json($clientSecret);

    const appearance = { theme: 'stripe', variables: { colorPrimary: '#0f172a' } };
    const elements = stripe.elements({ clientSecret, appearance });

    const linkAuthElement = elements.create('linkAuthentication');
    linkAuthElement.mount('#link-authentication-element');

    const paymentElement = elements.create('payment');
    paymentElement.mount('#payment-element');

    const form = document.getElementById('elements-form');
    const submitBtn = document.getElementById('elements-submit');
    const errorEl = document.getElementById('elements-error');

    form.addEventListener('submit', async (e) => {
      e.preventDefault();
      if (submitBtn.dataset.loading === '1') return;
      submitBtn.dataset.loading = '1';
      submitBtn.disabled = true;
      const original = submitBtn.textContent;
      submitBtn.textContent = submitBtn.getAttribute('data-loading-text') || 'Processing…';

      const email = document.getElementById('payer_email').value;

      const { error } = await stripe.confirmPayment({
        elements,
        confirmParams: {
          return_url: @json(route('payments.pay.return', $payment)),
          receipt_email: email || undefined,
        },
      });

      if (error) {
        errorEl.textContent = error.message || 'Something went wrong.';
        submitBtn.disabled = false;
        submitBtn.textContent = original;
        submitBtn.dataset.loading = '0';
      }
    });

    // Simple tab toggle
    document.querySelectorAll('.js-tab').forEach(btn => {
      btn.addEventListener('click', () => {
        document.querySelectorAll('.js-tab').forEach(b => b.setAttribute('aria-pressed', 'false'));
        btn.setAttribute('aria-pressed', 'true');
        const target = btn.dataset.target;
        document.querySelectorAll('.tabpanel').forEach(p => p.style.display = (p.id === target.substring(1)) ? '' : 'none');
      });
    });
  })();
</script>
@else
<script>
  // Tab toggle still needed if only Checkout is present (no Elements)
  document.querySelectorAll('.js-tab')?.forEach(btn => {
    btn.addEventListener('click', () => {
      document.querySelectorAll('.js-tab').forEach(b => b.setAttribute('aria-pressed', 'false'));
      btn.setAttribute('aria-pressed', 'true');
      const target = btn.dataset.target;
      document.querySelectorAll('.tabpanel').forEach(p => p.style.display = (p.id === target.substring(1)) ? '' : 'none');
    });
  });
</script>
@endif
@endsection
