{{-- resources/views/pay/show.blade.php --}}
<!doctype html>
<html lang="en">
<head>
  <meta charset="utf-8">
  <title>Complete Secure Payment</title>
  <meta name="viewport" content="width=device-width, initial-scale=1">

  <meta name="csrf-token" content="{{ csrf_token() }}">

  @php
    // ---- Brand / Terms detection (override by passing $brand) ----
    $host = request()->getHost();
    $brand = $brand ?? (str_contains($host, 'jimny') ? 'jimny' : 'dreamdrives');

    $brandName = $brand === 'jimny' ? 'Jimny Rentals' : 'Dream Drives';
    $logoPath  = $brand === 'jimny' ? '/images/jimny-logo.svg' : '/images/dreamdrives-logo.svg';
    $accent    = $brand === 'jimny' ? '#16a34a' : '#2563eb';   // green vs blue
    $accent600 = $brand === 'jimny' ? '#15803d' : '#1d4ed8';
    $termsUrl  = $brand === 'jimny'
        ? 'https://rental.jimny.co.nz/terms.html'
        : 'https://www.dreamdrives.co.nz/terms.html';
  @endphp

  <style>
    :root{
      --bg:#0b1727;
      --card:#111a2b;
      --border:#1b2a44;
      --text:#e9eef5;
      --muted:#9fb1d1;
      --accent: {{ $accent }};
      --accent-600: {{ $accent600 }};
      --success:#16a34a;
      --danger:#ef4444;
    }
    *{ box-sizing: border-box; }
    body{
      margin:0; background:var(--bg); color:var(--text);
      font-family: ui-sans-serif, system-ui, -apple-system, Segoe UI, Roboto, Helvetica, Arial, "Apple Color Emoji","Segoe UI Emoji";
    }
    .wrap{ max-width: 960px; margin: 2rem auto; padding: 1rem; }
    .header{ text-align:center; margin: 0 0 1rem; }
    .header img{ height: 42px; }
    .header h1{ margin: .5rem 0 0; font-weight: 700; }
    .header p{ margin:.25rem 0 0; color: var(--muted); }

    .grid{ display:grid; grid-template-columns: 1fr 1fr; gap: 1rem; }
    @media (max-width: 860px){ .grid{ grid-template-columns: 1fr; } }

    .card{
      background:var(--card);
      border:1px solid var(--border);
      border-radius:16px;
      padding:1.25rem;
    }
    .summary h2{ margin:0 0 .5rem; font-size:1.1rem; }
    .row{ display:flex; justify-content:space-between; gap:1rem; margin:.35rem 0; color:#cfd7e6; }
    .note{ margin:.75rem 0 0; color: var(--muted); font-size:.95rem; }
    .badges{ margin-top:.35rem; display:flex; gap:.4rem; flex-wrap:wrap; }
    .badge{ display:inline-block; font-size:.8rem; padding:.2rem .45rem; border-radius:999px; border:1px solid var(--border); color:#cfd7e6; }

    label{ display:block; font-size:.95rem; margin:.6rem 0 .35rem; color:#cfd7e6; }
    .text{
      width:100%; background:#0a1322; color:var(--text);
      border:1px solid var(--border); border-radius:10px; padding:.7rem .8rem; outline:none;
    }
    .text::placeholder{ color:#6f7f99; }

    .pay-card{ margin-top:.5rem; padding:.75rem .85rem; background:#0a1322; border:1px solid var(--border); border-radius:12px; }

    .btn{
      display:inline-flex; align-items:center; justify-content:center; gap:.5rem;
      margin-top:1rem; width:100%; padding:.95rem 1rem; border-radius:12px;
      background:var(--accent); color:#fff; font-weight:700; border:none; cursor:pointer;
      transition: background .15s ease, transform .02s ease;
    }
    .btn:hover{ background:var(--accent-600); }
    .btn:active{ transform: translateY(1px); }
    .btn[disabled]{ opacity:.6; cursor:not-allowed; }

    .spinner{
      width:18px; height:18px; border-radius:50%;
      border:2px solid rgba(255,255,255,.28); border-top-color:#fff;
      animation:spin .8s linear infinite;
    }
    @keyframes spin{ to{ transform:rotate(360deg); } }

    .trust{ margin-top:1rem; display:flex; align-items:center; gap:.75rem; flex-wrap:wrap; }
    .trust img{ height:22px; opacity:.9; filter: grayscale(15%); }
    .trust p{ margin:.25rem 0 0; font-size:.85rem; color:var(--muted); width:100%; }

    .alert{
      margin:.75rem 0 0; background:#0a1322; border:1px solid var(--border);
      border-left:4px solid var(--success); padding:.6rem .8rem; border-radius:10px;
      color:#d7ffe1; font-size:.95rem;
    }
    .error{
      margin:.75rem 0 0; background:#1b0f12; border:1px solid #4c1d1d; border-left:4px solid var(--danger);
      color:#fecaca; padding:.6rem .8rem; border-radius:10px; display:none;
    }
    .help{ margin-top:.75rem; color:var(--muted); font-size:.9rem; }
    a.link{ color:#cbd5ff; text-decoration:underline; text-underline-offset:2px; }
  </style>

  <script src="https://js.stripe.com/v3/"></script>
</head>
<body>
  <div class="wrap">

    <!-- Branding -->
    <div class="header">
      <img src="{{ $logoPath }}" alt="{{ $brandName }}" onerror="this.style.display='none'">
      <h1>Complete Secure Payment</h1>
      <p>Jimny Rentals and Dream Drives</p>
    </div>

    <div class="grid">
      <!-- Left: Payment form -->
      <div class="card">
        <form id="pay-form" novalidate>
          <input type="hidden" id="token" value="{{ $token }}">

          <label for="payer_name">Name on card</label>
          <input id="payer_name" name="payer_name" class="text" type="text" placeholder="Jane Appleseed" autocomplete="name">

          <label for="receipt_email">Receipt email</label>
          <input id="receipt_email" name="receipt_email" class="text" type="email" placeholder="you@example.com" autocomplete="email">

          <label>Card details</label>
          <div id="payment-element" class="pay-card"></div>

          <div id="error" class="error"></div>

          <button id="pay-button" class="btn" type="submit">
            <span class="btn-text">{{ $isBondHold ? 'Authorise bond hold' : 'Pay now' }}</span>
            <span class="spinner" style="display:none;"></span>
          </button>

          <div class="trust">
            <img src="/images/powered-by-stripe.svg" alt="Powered by Stripe" onerror="this.style.display='none'">
            <img src="/images/visa.svg" alt="Visa" onerror="this.style.display='none'">
            <img src="/images/mastercard.svg" alt="Mastercard" onerror="this.style.display='none'">
            <img src="/images/amex.svg" alt="American Express" onerror="this.style.display='none'">
            <p>Payments are securely processed by Stripe. Your card details are encrypted and never stored on our servers.</p>
          </div>

          <p class="help">
            By completing this {{ $isBondHold ? 'authorisation' : 'payment' }} you agree to
            <a class="link" href="{{ $termsUrl }}" target="_blank" rel="noopener">our terms</a>.
            Need help? Email <a class="link" href="mailto:info@dreamdrives.co.nz">info@dreamdrives.co.nz</a>.
          </p>
        </form>
      </div>

      <!-- Right: Summary & reassurance -->
      <div class="card summary">
  <h2>Booking Summary</h2>

  <div class="row"><span>Booking Reference</span><strong>{{ $reference ?? '—' }}</strong></div>

  @if($startDate || $endDate)
    <div class="row">
      <span>Booking dates</span>
      <strong>
        {{ $startDate ?? '—' }} → {{ $endDate ?? '—' }}
      </strong>
    </div>
  @endif

  <div class="row">
    <span>{{ $isBondHold ? 'Hold amount' : 'Amount due' }}</span>
<strong>${{ number_format($amountCents / 100, 2) }} {{ strtoupper($currency ?? 'NZD') }}</strong>
  </div>

  @if($isBondHold)
    <div class="alert">
      A refundable <strong>bond hold</strong> will be authorised on your card — not charged.
      It will be released automatically once your rental is completed and any post-hire charges are settled.
    </div>
  @else
    <p class="note">Your card is charged immediately. A receipt will be emailed to the address you provide.</p>
  @endif

  <div class="badges">
    <span class="badge">TLS 1.2+</span>
    <span class="badge">PCI DSS</span>
    <span class="badge">3-D Secure</span>
  </div>
      </div>
    </div>
  </div>

  <script>
    // Autofocus for quick entry
    setTimeout(()=> document.getElementById('payer_name')?.focus(), 150);

    const stripe = Stripe(@json($stripePk));
    const returnUrl = @json(route('payments.public.done', ['token' => $token]));
    const sessionUrl = @json(route('payments.public.session')); // POST; returns {clientSecret}

    const form   = document.getElementById('pay-form');
    const payBtn = document.getElementById('pay-button');
    const spinner= payBtn.querySelector('.spinner');
    const btnTxt = payBtn.querySelector('.btn-text');
    const errorBox = document.getElementById('error');

    let elements;

    function setLoading(on){
      payBtn.disabled = on;
      spinner.style.display = on ? 'inline-block' : 'none';
      btnTxt.textContent = on ? 'Processing…' : btnTxt.getAttribute('data-label') || btnTxt.textContent;
    }

    function showError(msg){
      errorBox.textContent = msg || 'Something went wrong. Please try again.';
      errorBox.style.display = 'block';
      setTimeout(()=> errorBox.scrollIntoView({behavior:'smooth', block:'center'}), 50);
    }

    async function initElements() {
      // Create/retrieve PI via your session endpoint
      const body = {
        token: document.getElementById('token').value,
        receipt_email: document.getElementById('receipt_email').value || null,
        payer_name: document.getElementById('payer_name').value || null,
      };

      const res = await fetch(sessionUrl, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content
        },
        body: JSON.stringify(body)
      });

      const data = await res.json();
      if (!res.ok || !data.clientSecret) {
        throw new Error(data.error || 'Could not initialise payment form.');
      }

      elements = stripe.elements({
        clientSecret: data.clientSecret,
        appearance: { theme: 'night' }
      });

      const paymentElement = elements.create('payment', { terms: { card: 'always' } });
      paymentElement.mount('#payment-element');
    }

    form.addEventListener('submit', async (e) => {
      e.preventDefault();
      errorBox.style.display = 'none';
      setLoading(true);

      const email = document.getElementById('receipt_email').value || undefined;
      const name  = document.getElementById('payer_name').value || undefined;

      const { error } = await stripe.confirmPayment({
        elements,
        confirmParams: {
          return_url: returnUrl,
          payment_method_data: {
            billing_details: { name, email }
          }
        }
      });

      if (error) {
        showError(error.message);
        setLoading(false);
      }
    });

    // Kick off
    initElements().catch(err => {
      console.error(err);
      showError(err.message);
    });
  </script>
</body>
</html>
