<!doctype html>
<html lang="en">
<head>
  <meta charset="utf-8">
  <title>{{ $title ?? 'Secure payment' }}</title>
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <meta name="csrf-token" content="{{ csrf_token() }}">
  <script src="https://js.stripe.com/v3/"></script>
  <style>
    body{font-family:ui-sans-serif,system-ui;margin:24px;background:#f8fafc}
    .card{max-width:520px;margin:10vh auto;padding:24px;border:1px solid #eee;border-radius:16px;background:#fff}
    .muted{color:#6b7280}
    .row{margin-top:12px}
    button{padding:.75rem 1rem;border-radius:.5rem;background:#4f46e5;color:#fff;border:0;cursor:pointer;font-weight:600;width:100%}
    button[disabled]{opacity:.6;cursor:not-allowed}
    .border{border:1px solid #e5e7eb;border-radius:8px;padding:12px;background:#fff}
    .hidden{display:none}
    .h1{font-size:1.5rem;font-weight:700;margin:0 0 4px}
  </style>
</head>
<body>
  @php
    // ----- Inputs expected from controller -----
    // $mode: 'none' | 'charge' | 'hold' | 'charge+hold'
    // $job:  ['id' => int, 'ref' => ?string, 'start_at' => ?string, 'end_at' => ?string] (optional)
    // $amount_due_cents: int
    // $hold_cents: int
    // $clientSecret: ?string  (present for hold-only)
    // $brand: string
    // $currency: string like 'nzd'
    // $reference: ?string
    $mode = $mode ?? 'none';
    $amountDueCents = (int)($amount_due_cents ?? 0);
    $holdCents = (int)($hold_cents ?? 0);
    $clientSecretFromServer = $clientSecret ?? null; // used for hold-only
    $currency = strtolower($currency ?? 'nzd');
    $brand = $brand ?? config('app.name');
    $publishableKey = $stripePk ?? config('services.stripe.key') ?? env('STRIPE_KEY');
    $job = $job ?? null;
  @endphp

  <div class="card">
    <h1 class="h1" id="header"></h1>

    @if(!empty($reference))
      <p class="muted">Reference: <strong>{{ $reference }}</strong></p>
    @endif

    <div id="summary" class="row border">
      @if($holdCents > 0)
        <div><strong>Refundable security hold:</strong> NZ${{ number_format($holdCents/100, 2) }} (manual capture)</div>
      @endif
      @if($amountDueCents > 0)
        <div class="row" style="margin-top:6px">
          <strong>Amount due now:</strong> NZ${{ number_format($amountDueCents/100, 2) }}
        </div>
      @endif
      @if($amountDueCents <= 0 && $holdCents <= 0)
        <div class="muted">No charges or holds required.</div>
      @endif
    </div>

    <div id="nothing" class="row muted hidden">
      There’s nothing due for this booking.
    </div>

    <div id="payment-element" class="row hidden"></div>
    <button id="submit" class="row hidden">Continue</button>
    <div id="msg" class="row muted"></div>
  </div>

  <script>
    // ---- Blade → JS bridges ----
    const MODE = @json($mode);
    const PUBLISHABLE_KEY = @json($publishableKey ?? null);
    const CLIENT_SECRET_FROM_SERVER = @json($clientSecretFromServer ?? null);
    const CURRENCY = @json($currency);
    const BRAND = @json($brand);
    const AMOUNT_DUE_CENTS = @json($amountDueCents);
    const HOLD_CENTS = @json($holdCents);
    const JOB = @json($job);
    const CSRF = document.querySelector('meta[name="csrf-token"]')?.getAttribute('content');

    // ---- DOM helpers ----
    const el = (id) => document.getElementById(id);
    const headerEl = el('header');
    const msgEl = el('msg');
    const nothingEl = el('nothing');
    const paymentElementContainer = el('payment-element');
    const submitBtn = el('submit');

    function money(nzdCents) {
      return 'NZ$' + (nzdCents/100).toFixed(2);
    }
    function setHeader() {
      let text = 'Nothing to pay';
      if (MODE === 'hold') text = 'Authorise your refundable security hold';
      else if (MODE === 'charge+hold') text = 'Complete your payment & authorise hold';
      else if (MODE === 'charge') text = 'Complete your payment';
      headerEl.textContent = text;
    }
    function show(elm) { elm.classList.remove('hidden'); }
    function hide(elm) { elm.classList.add('hidden'); }
    function note(s)  { msgEl.textContent = s || ''; }

    // ---- Stripe setup ----
    let stripe, elements;

    async function ensureClientSecret() {
      // For hold-only, we expect the client secret passed in directly.
      if (MODE === 'hold') {
        return CLIENT_SECRET_FROM_SERVER || null;
      }

      // For 'charge' or 'charge+hold', fetch client secret from server
      // Adjust the endpoint/payload to your API as needed.
      try {
        const res = await fetch('/api/payment_intents', {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json',
            ...(CSRF ? {'X-CSRF-TOKEN': CSRF} : {})
          },
          body: JSON.stringify({
            mode: MODE,
            job_id: JOB?.id ?? null,
            amount_due_cents: AMOUNT_DUE_CENTS,
            hold_cents: HOLD_CENTS,
            currency: CURRENCY,
            brand: BRAND,
          })
        });

        if (!res.ok) {
          const t = await res.text();
          throw new Error('Failed to create PaymentIntent: ' + t);
        }

        const data = await res.json();
        // Expect data like: { client_secret: 'pi_..._secret_...' }
        return data.client_secret ?? null;
      } catch (e) {
        console.error(e);
        note('Could not initialize payment. Please try again or contact support.');
        return null;
      }
    }

    function submitLabel() {
      if (MODE === 'hold') return `Authorise ${money(HOLD_CENTS)} hold`;
      if (MODE === 'charge+hold') return 'Pay & Authorise Hold';
      if (MODE === 'charge') return 'Pay Now';
      return 'Continue';
    }

    async function mountElements(clientSecret) {
      if (!PUBLISHABLE_KEY) {
        note('Missing Stripe publishable key.');
        return;
      }

      stripe = Stripe(PUBLISHABLE_KEY);

      if (!clientSecret) {
        note('Missing client secret.');
        return;
      }

      elements = stripe.elements({ clientSecret, appearance: { theme: 'stripe' }});
      const paymentElement = elements.create('payment');
      paymentElement.mount('#payment-element');

      // Show UI
      show(paymentElementContainer);
      submitBtn.textContent = submitLabel();
      show(submitBtn);
      note('');
    }

    async function init() {
      setHeader();

      // Gate form rendering:
      const shouldRenderForm =
        MODE === 'hold' || MODE === 'charge' || MODE === 'charge+hold';

      if (!shouldRenderForm) {
        show(nothingEl);
        return;
      }

      note('Initialising…');

      const clientSecret = await ensureClientSecret();
      if (!clientSecret) {
        // Error already shown
        return;
      }

      await mountElements(clientSecret);
      note('');
    }

    // ---- submit handler ----
    let submitting = false;
    submitBtn.addEventListener('click', async () => {
      if (submitting || !stripe || !elements) return;
      submitting = true;
      submitBtn.disabled = true;
      note('Processing…');

      try {
        const result = await stripe.confirmPayment({
          elements,
          // If you prefer a redirect flow, change to `redirect: 'always'`
          redirect: 'if_required',
          confirmParams: {
            // If using a redirect success page:
            return_url: "{{ url('/pay/done') }}",
          },
        });

        submitting = false;
        submitBtn.disabled = false;

        if (result.error) {
          note(result.error.message || 'Payment failed.');
          return;
        }

        // If no error and no redirect, success is implied. Navigate to your success page.
        window.location.href = "/p/success";
      } catch (e) {
        submitting = false;
        submitBtn.disabled = false;
        console.error(e);
        note('Something went wrong while confirming the payment.');
      }
    });

    init();
  </script>
</body>
</html>
