@php
    /** @var \App\Models\Owner $owner */
    /** @var \Illuminate\Support\Carbon $from */
    /** @var \Illuminate\Support\Carbon $to */
    /** @var \Illuminate\Support\Collection $entries */
    /** @var array $totals */
    /** @var string $statementId */
    /** @var int $payoutAmountCents */
    /** @var \Illuminate\Support\Collection|\Illuminate\Support\Collection[]|array $vehicles */
    /** @var array|null $previewTotals */
    /** @var \Illuminate\Support\Carbon|null $previewPayoutDate */
    /** @var \Illuminate\Support\Collection|null $previewJobs */

    // Normalise preview variables so the view never explodes
    $previewTotals     = $previewTotals     ?? ['owner_total_cents' => 0, 'month_label' => null, 'per_vehicle_cents' => []];
    $previewPayoutDate = $previewPayoutDate ?? now()->copy()->day(10);
    $previewJobs       = $previewJobs       ?? collect();

    $earnings    = $totals['earnings_cents']    ?? 0;
    $costs       = $totals['costs_cents']       ?? 0;
    $payouts     = $totals['payouts_cents']     ?? 0;
    $adjustments = $totals['adjustments_cents'] ?? 0;
    $balance     = $totals['balance_cents']     ?? 0;

    $fmt = fn (int $cents) => '$' . number_format($cents / 100, 2);

    $today = now();

    $payoutReference = $owner->bank_reference
        ?: ('DD-' . $statementId);

    // Per-vehicle breakdown for the current statement period
    $vehicleSummary = [];
    foreach ($entries as $entry) {
        $name = $entry->vehicle->name ?? 'Unassigned';

        $typeEnum = $entry->type;
        $type = $typeEnum instanceof \BackedEnum ? $typeEnum->value : (string) $typeEnum;

        $rawCents     = (int) $entry->amount_cents;
        $ownerPercent = $entry->vehicle?->owner_share_percent;

        $ownerCents = $rawCents;
        if ($type === 'earning' && $ownerPercent !== null) {
            $ownerCents = (int) round($rawCents * ($ownerPercent / 100));
        } elseif (in_array($type, ['cost', 'payout'], true)) {
            $ownerCents = $rawCents;
        }

        if (! isset($vehicleSummary[$name])) {
            $vehicleSummary[$name] = [
                'gross_cents' => 0,
                'owner_cents' => 0,
            ];
        }

        if ($type === 'earning') {
            $vehicleSummary[$name]['gross_cents'] += $rawCents;
            $vehicleSummary[$name]['owner_cents'] += $ownerCents;
        }
    }

    // Preview totals (next payout)
    $previewOwnerTotal = $previewTotals['owner_total_cents'] ?? 0;
    $previewMonthLabel = $previewTotals['month_label'] ?? $previewPayoutDate->format('F Y');
    $previewPerVehicle = $previewTotals['per_vehicle_cents'] ?? [];
@endphp

<!doctype html>
<html>
<head>
    <meta charset="utf-8">
    <title>Owner Statement</title>
    <style>
        body {
            font-family: sans-serif;
            font-size: 12px;
            color: #111827;
            margin: 0;
            padding: 24px;
            background: #f9fafb;
        }
        .container {
            max-width: 900px;
            margin: 0 auto;
            background: #ffffff;
            border-radius: 8px;
            padding: 24px 28px 28px;
            box-shadow: 0 4px 14px rgba(15, 23, 42, 0.08);
        }
        h1 {
            font-size: 22px;
            margin: 0 0 4px;
        }
        h2 {
            font-size: 14px;
            margin: 18px 0 4px;
        }
        .subheading {
            font-size: 11px;
            color: #6b7280;
            margin-bottom: 16px;
        }
        .meta-table {
            width: 100%;
            border-collapse: collapse;
            margin-bottom: 12px;
        }
        .meta-table td {
            padding: 2px 0;
            font-size: 11px;
            vertical-align: top;
        }
        .meta-label {
            color: #6b7280;
            width: 110px;
        }
        .muted {
            color: #6b7280;
            font-size: 11px;
        }
        .pill {
            display: inline-block;
            padding: 2px 8px;
            border-radius: 999px;
            background: #eef2ff;
            color: #4338ca;
            font-size: 10px;
            font-weight: 600;
        }

        .panel {
            border-radius: 8px;
            border: 1px solid #e5e7eb;
            background: #f9fafb;
            padding: 10px 12px;
            margin-bottom: 14px;
        }
        .panel-title {
            font-size: 11px;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.04em;
            color: #6b7280;
            margin-bottom: 4px;
        }

        table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 6px;
        }
        th, td {
            border: 1px solid #e5e7eb;
            padding: 4px 6px;
            text-align: left;
        }
        th {
            background: #f3f4f6;
            font-size: 11px;
        }
        td {
            font-size: 11px;
        }
        .text-right { text-align: right; }
        .row-highlight {
            background: #f9fafb;
            font-weight: 600;
        }
        .ledger-header {
            margin-top: 18px;
        }
        .ledger-table tbody tr:nth-child(even) {
            background: #fafafa;
        }
    </style>
</head>
<body>
<div class="container">
    <h1>Owner Statement</h1>
    <div class="subheading">
        Wilberforce Offroad Ltd T/A Dream Drives
    </div>

    <table class="meta-table">
        <tr>
            <td class="meta-label">Company GST</td>
            <td>{{ \App\Models\Owner::COMPANY_GST_NUMBER }}</td>
            <td class="meta-label">Owner</td>
            <td>{{ $owner->legal_name ?? $owner->name }}</td>
        </tr>
        <tr>
            <td class="meta-label">Owner GST</td>
            <td>{{ $owner->gst_registered ? ($owner->gst_number ?: '—') : 'Not registered' }}</td>
            <td class="meta-label">Period</td>
            <td>{{ $from->toDateString() }} – {{ $to->toDateString() }}</td>
        </tr>
        <tr>
            <td class="meta-label">Generated</td>
            <td>{{ $today->toDateTimeString() }}</td>
            <td class="meta-label">Statement ID</td>
            <td>
                <span class="pill">
                    {{ $statementId }}
                </span>
            </td>
        </tr>
    </table>

    <p class="muted" style="margin-bottom: 12px;">
        Amounts shown are inclusive of GST where applicable.
        @if ($owner->gst_registered)
            This statement can be used to support your GST records.
        @else
            This statement is not a tax invoice.
        @endif
    </p>

    <div class="panel">
        <div class="panel-title">Payment</div>
        <table style="border-collapse: collapse; width: 100%; border: none; margin-top: 0;">
            <tr>
                <td style="border: none; padding: 2px 0;">Payout for this statement</td>
                <td class="text-right" style="border: none; padding: 2px 0;">
                    <strong>{{ $fmt($payoutAmountCents) }}</strong>
                </td>
            </tr>
            <tr>
                <td style="border: none; padding: 2px 0;">Paid on</td>
                <td class="text-right" style="border: none; padding: 2px 0;">
                    <strong>{{ $today->toDateString() }}</strong>
                </td>
            </tr>
            <tr>
                <td style="border: none; padding: 2px 0;">Paid to</td>
                <td class="text-right" style="border: none; padding: 2px 0;">
                    {{ $owner->bank_account_name ?: ($owner->legal_name ?? $owner->name) }}
                    @if ($owner->bank_account_number)
                        · {{ $owner->bank_account_number }}
                    @endif
                </td>
            </tr>
            <tr>
                <td style="border: none; padding: 2px 0;">Bank reference</td>
                <td class="text-right" style="border: none; padding: 2px 0;">
                    {{ $payoutReference }}
                </td>
            </tr>
        </table>
    </div>

    @if(!empty($vehicleSummary))
        <h2>Vehicle summary</h2>
        <table>
            <thead>
                <tr>
                    <th>Vehicle</th>
                    <th class="text-right">Gross earnings</th>
                    <th class="text-right">Owner share</th>
                </tr>
            </thead>
            <tbody>
                @foreach ($vehicleSummary as $vehicleName => $row)
                    <tr>
                        <td>{{ $vehicleName }}</td>
                        <td class="text-right">{{ $fmt($row['gross_cents']) }}</td>
                        <td class="text-right">{{ $fmt($row['owner_cents']) }}</td>
                    </tr>
                @endforeach
            </tbody>
        </table>
    @endif

    <h2>Summary</h2>
    <table>
        <tr>
            <td>Earnings (your share)</td>
            <td class="text-right"><strong>{{ $fmt($earnings) }}</strong></td>
        </tr>
        <tr>
            <td>Adjustments</td>
            <td class="text-right">{{ $fmt($adjustments) }}</td>
        </tr>
        <tr>
            <td>Costs</td>
            <td class="text-right">{{ $fmt($costs) }}</td>
        </tr>
        <tr>
            <td>Payouts</td>
            <td class="text-right">{{ $fmt($payouts) }}</td>
        </tr>
        <tr class="row-highlight">
            <td>Balance for this period</td>
            <td class="text-right">{{ $fmt($balance) }}</td>
        </tr>
    </table>

    @if ($previewOwnerTotal > 0)
        <h2>Future earnings (next payout)</h2>
        <p class="muted" style="margin-bottom: 4px;">
            Estimated owner share for bookings ending in {{ $previewMonthLabel }}.
            These will be paid on {{ $previewPayoutDate->toDateString() }}
            according to our payout schedule (10th of each month for the previous month’s completed bookings).
        </p>
        <table>
            <thead>
                <tr>
                    <th>Vehicle</th>
                    <th class="text-right">Estimated owner share</th>
                </tr>
            </thead>
            <tbody>
                @foreach ($previewPerVehicle as $vehicleName => $ownerCents)
                    <tr>
                        <td>{{ $vehicleName }}</td>
                        <td class="text-right">{{ $fmt($ownerCents) }}</td>
                    </tr>
                @endforeach
                <tr class="row-highlight">
                    <td>Total estimated owner share</td>
                    <td class="text-right">{{ $fmt($previewOwnerTotal) }}</td>
                </tr>
            </tbody>
        </table>
    @endif

    @if ($previewJobs->isNotEmpty())
        <h2>Upcoming bookings</h2>
        <table>
            <thead>
                <tr>
                    <th>Vehicle</th>
                    <th>Start</th>
                    <th>End</th>
                    <th class="text-right">Est. owner share</th>
                </tr>
            </thead>
            <tbody>
                @foreach ($previewJobs as $job)
                    @php
                        $ownerPercent = $job->vehicle?->owner_share_percent ?? 100;
                        $gross        = (int) ($job->charge_amount_cents ?? $job->total_price_cents ?? 0);
                        $ownerCents   = (int) round($gross * ($ownerPercent / 100));
                    @endphp
                    <tr>
                        <td>{{ $job->vehicle?->name }}</td>
                        <td>{{ \Illuminate\Support\Carbon::parse($job->start_at)->toDateString() }}</td>
                        <td>{{ \Illuminate\Support\Carbon::parse($job->end_at)->toDateString() }}</td>
                        <td class="text-right">{{ $fmt($ownerCents) }}</td>
                    </tr>
                @endforeach
            </tbody>
        </table>
    @endif

    <h2 class="ledger-header">Ledger entries</h2>
    <table class="ledger-table">
        <thead>
            <tr>
                <th>Date</th>
                <th>Type</th>
                <th class="text-right">Raw amount</th>
                <th class="text-right">Owner %</th>
                <th class="text-right">Owner amount</th>
                <th>Vehicle</th>
                <th>Booking</th>
                <th>Description</th>
            </tr>
        </thead>
        <tbody>
        @forelse ($entries as $entry)
            @php
                $typeEnum = $entry->type;
                $type = $typeEnum instanceof \BackedEnum ? $typeEnum->value : (string) $typeEnum;

                $rawCents     = (int) $entry->amount_cents;
                $ownerPercent = $entry->vehicle?->owner_share_percent;

                $ownerCents = $rawCents;

                if ($type === 'earning' && $ownerPercent !== null) {
                    $ownerCents = (int) round($rawCents * ($ownerPercent / 100));
                } elseif (in_array($type, ['cost', 'payout'], true)) {
                    $ownerCents = $rawCents;
                }

                $date = $entry->occurred_at
                    ? \Illuminate\Support\Carbon::parse($entry->occurred_at)->format('Y-m-d H:i')
                    : '';
            @endphp
            <tr>
                <td>{{ $date }}</td>
                <td>{{ ucfirst($type) }}</td>
                <td class="text-right">{{ $fmt($rawCents) }}</td>
                <td class="text-right">
                    @if ($type === 'earning' && $ownerPercent !== null)
                        {{ $ownerPercent }}%
                    @elseif (in_array($type, ['cost', 'payout'], true))
                        100%
                    @else
                        —
                    @endif
                </td>
                <td class="text-right">{{ $fmt($ownerCents) }}</td>
                <td>{{ $entry->vehicle?->name }}</td>
                <td>{{ $entry->job?->booking_reference }}</td>
                <td>{{ $entry->description }}</td>
            </tr>
        @empty
            <tr>
                <td colspan="8">No entries for this period.</td>
            </tr>
        @endforelse
        </tbody>
    </table>

    <p class="muted" style="margin-top: 14px;">
        If anything in this statement doesn’t look right, please contact us as soon as possible.
    </p>
</div>
</body>
</html>
