<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <title>Security Hold</title>
  <meta name="viewport" content="width=device-width, initial-scale=1" />
  <script src="https://js.stripe.com/v3/"></script>
  <style>
    :root {
      --ink:#0f172a; --muted:#64748b; --ok:#16a34a; --err:#dc2626; --cta:#2563eb; --cta-ink:#fff;
      --border:#e5e7eb; --bg:#f8fafc; --card:#fff; --radius:12px;
    }
    *{box-sizing:border-box}
    body{margin:0;background:var(--bg);color:var(--ink);font:16px/1.5 system-ui,-apple-system,Segoe UI,Roboto,Helvetica,Arial,sans-serif}
    .container{max-width:480px;margin:40px auto;padding:24px;background:var(--card);border:1px solid var(--border);border-radius:var(--radius);box-shadow:0 6px 16px rgba(2,6,23,.06)}
    h1{margin:0 0 8px;font-size:1.25rem}
    p.lead{margin:0 0 16px;color:var(--muted)}
    .row{display:grid;gap:12px;margin-top:12px}
    label{font-size:.9rem}
    input[type="email"], input[type="text"]{width:100%;padding:.7rem .8rem;border:1px solid var(--border);border-radius:10px}
    #card-element{padding:.8rem;border:1px solid var(--border);border-radius:10px;background:#fff}
    .hint{font-size:.85rem;color:var(--muted)}
    .cta{margin-top:14px;display:block;width:100%;padding:.9rem 1rem;border:0;border-radius:12px;background:var(--cta);color:var(--cta-ink);font-weight:600;cursor:pointer;transition:transform .02s ease}
    .cta[disabled]{opacity:.6;cursor:not-allowed}
    .cta:active{transform:translateY(1px)}
    .status{margin-top:12px;font-size:.92rem}
    .status.ok{color:var(--ok)}
    .status.err{color:var(--err)}
    .banner{padding:.6rem .8rem;border-radius:10px;margin:10px 0;background:#fef2f2;color:#991b1b;border:1px solid #fecaca;display:none}
    .flex{display:flex;gap:10px;align-items:center}
    .spinner{display:inline-block;width:16px;height:16px;border:2px solid #fff;border-right-color:transparent;border-radius:50%;animation:spin .6s linear infinite;vertical-align:-3px}
    @keyframes spin{to{transform:rotate(360deg)}}
  </style>
</head>
<body>
  <div class="container">
    <h1>Security Bond Hold</h1>
    <p class="lead">We’ll place a temporary hold on your card (no money is taken now). Your card will also be securely saved for permitted off-session adjustments.</p>

    <!-- Optional top-level warning slot (hidden by default) -->
    <div id="banner" class="banner"></div>

    <form id="hold-form" autocomplete="on">
      <div class="row">
        <div>
          <label for="cardholder">Name on card</label>
          <input id="cardholder" name="cardholder" type="text" inputmode="text" autocomplete="cc-name" placeholder="Jane Appleseed" />
        </div>
        <div>
          <label for="receiptEmail">Receipt email (optional)</label>
          <input id="receiptEmail" name="receiptEmail" type="email" inputmode="email" autocomplete="email" placeholder="you@example.com" />
          <div class="hint">We’ll email a confirmation once the hold is placed.</div>
        </div>
      </div>

      <div class="row">
        <div>
          <label>Card details</label>
          <div id="card-element"><!-- Stripe CardElement mounts here --></div>
        </div>
      </div>

      <button id="submitBtn" class="cta" type="submit">
        <span class="flex" style="justify-content:center;">
          <span id="btnText">Place NZ$500 hold</span>
          <span id="btnSpin" class="spinner" style="display:none;"></span>
        </span>
      </button>

      <div id="status" class="status" aria-live="polite"></div>
    </form>
  </div>

  <script>
    // 1) Stripe init
    const stripe = Stripe("{{ config('services.stripe.key') }}");
    const elements = stripe.elements();
    const card = elements.create('card', { hidePostalCode: false });
    card.mount('#card-element');

    // 2) Helpers
    const $ = (sel) => document.querySelector(sel);
    const setBusy = (busy) => {
      $('#submitBtn').disabled = busy;
      $('#btnSpin').style.display = busy ? 'inline-block' : 'none';
    };
    const showBanner = (msg) => { const b = $('#banner'); b.textContent = msg; b.style.display = 'block'; };
    const clearBanner = () => { const b = $('#banner'); b.style.display = 'none'; b.textContent = ''; };

    // 3) Form submit → create PI (manual capture) → confirm
    $('#hold-form').addEventListener('submit', async (e) => {
      e.preventDefault();
      clearBanner();
      $('#status').textContent = '';
      setBusy(true);

      try {
        // 3a) Ask backend for a PaymentIntent (capture_method=manual, setup_future_usage=off_session)
        const res = await fetch("{{ route('portal.pay.hold.intent', ['token' => $booking->portal_token]) }}", {
          method: 'POST',
          credentials: 'same-origin',
          headers: {
            'Content-Type': 'application/json',
            'Accept': 'application/json',
            'X-CSRF-TOKEN': "{{ csrf_token() }}",
            'X-Requested-With': 'XMLHttpRequest'
          },
          body: JSON.stringify({
            // optional: forward email/name to server if you want them on the PI
            receipt_email: $('#receiptEmail').value || null,
            cardholder: $('#cardholder').value || null
          })
        });

        if (!res.ok) {
          let t = 'Could not start the hold.';
          try { const j = await res.json(); if (j && j.error) t = j.error; } catch(_) {}
          throw new Error(t);
        }
        const payload = await res.json();
        const clientSecret = payload.client_secret || payload.clientSecret;
        if (!clientSecret) throw new Error('Server did not return a client_secret for the hold.');

        // 3b) Confirm the PaymentIntent (places the hold; may trigger 3DS)
        const confirmOpts = {
          payment_method: { card, billing_details: { name: $('#cardholder').value || undefined, email: $('#receiptEmail').value || undefined } }
        };

        const { error, paymentIntent } = await stripe.confirmCardPayment(clientSecret, confirmOpts);

        if (error) {
          $('#status').textContent = error.message || 'Payment failed.';
          $('#status').className = 'status err';
          setBusy(false);
          return;
        }

        // 3c) Success path: for holds we expect requires_capture
        if (paymentIntent && (paymentIntent.status === 'requires_capture' || paymentIntent.status === 'succeeded')) {
          $('#status').textContent = 'Hold placed successfully.';
          $('#status').className = 'status ok';

          // Optional: tell your backend to persist the PI id / mark deposit state
          // You may already handle this in your webhook (preferred).
          window.location.href = "{{ route('portal.pay.hold.complete', ['token' => $booking->portal_token]) }}?payment_intent=" + encodeURIComponent(paymentIntent.id);
          return;
        }

        // Anything else is unexpected for a hold
        $('#status').textContent = 'Unexpected status: ' + (paymentIntent ? paymentIntent.status : 'unknown');
        $('#status').className = 'status err';
        setBusy(false);

      } catch (err) {
        $('#status').textContent = err.message || String(err);
        $('#status').className = 'status err';
        setBusy(false);
      }
    });
  </script>
</body>
</html>
