<?php

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Route;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;

use App\Http\Controllers\Webhooks\StripeWebhookController;
use App\Http\Controllers\Payments\WebhookController;
use App\Http\Controllers\VevsWebhookController;
use App\Services\VevsClient;
use App\Http\Controllers\Payments\PaymentIntentController;


/*
|--------------------------------------------------------------------------
| API Routes
|--------------------------------------------------------------------------
| Stateless JSON endpoints using the "api" middleware group.
| Versioned under /api/v1. Webhooks can live here (no CSRF).
|--------------------------------------------------------------------------
*/

Route::post('/payment_intents', [PaymentIntentController::class, 'store'])->name('payment_intents.store');


// --- Webhooks ---
Route::post('/integrations/vevs', [VevsWebhookController::class, 'handle'])
    ->name('integrations.vevs');

/**
 * Safer, logged, throttled pull route (staging only)
 */
Route::middleware('throttle:30,1')->get('/integrations/vevs/pull', function (Request $r) {
    abort_unless(app()->environment('staging'), 403, 'Disabled outside staging.');

    $rid = (string) ($r->header('X-Request-Id') ?: Str::uuid());
    $ip  = $r->ip();

    $secret = (string) config('services.internal.pull_secret', '');
    $header = (string) $r->header('X-Internal-Secret', '');
    abort_unless($secret !== '' && hash_equals($secret, $header), 403);

    $refId = trim((string) $r->query('ref_id', ''));
    abort_unless($refId !== '' && preg_match('/^[A-Za-z0-9_-]+$/', $refId), 422, 'Invalid ref_id');

    Log::info('[VEVS pull] start', ['rid' => $rid, 'ip' => $ip, 'ref_id' => $refId]);

    try {
        $raw = VevsClient::make()->getReservation($refId);

        // Optional: redact PII if ?sanitize=1
        if ($r->boolean('sanitize')) {
            if (class_exists(\App\Support\Sanitize\VevsReservation::class)) {
                $raw = \App\Support\Sanitize\VevsReservation::redact($raw);
            }
        }

        Log::info('[VEVS pull] ok', ['rid' => $rid, 'ref_id' => $refId]);

        return response()
            ->json(['ok' => true, 'data' => $raw], 200)
            ->header('X-Request-Id', $rid);
    } catch (\Throwable $e) {
        Log::warning('[VEVS pull] error', [
            'rid'    => $rid,
            'ref_id' => $refId,
            'msg'    => $e->getMessage(),
        ]);

        return response()
            ->json(['ok' => false, 'error' => 'VEVS_PULL_FAILED', 'message' => $e->getMessage()], 502)
            ->header('X-Request-Id', $rid);
    }
})->name('integrations.vevs.pull');

Route::options('/integrations/{any}', fn () => response()->noContent())
    ->where('any', '.*');

Route::post('/webhooks/stripe', [StripeWebhookController::class, 'handle'])
    ->name('webhooks.stripe.api');

Route::post('/webhooks/{provider}', [WebhookController::class, 'receive'])
    ->name('webhooks.receive');

/*
|--------------------------------------------------------------------------
| v1: Basic utilities
|--------------------------------------------------------------------------
*/
Route::prefix('v1')->group(function () {
    // Simple ping with timestamp & path
    Route::get('/ping', function () {
        return response()->json([
            'ok'   => true,
            'path' => '/api/v1/ping',
            'ts'   => now()->toIso8601String(),
        ]);
    })->name('api.v1.ping');

    // CORS preflight convenience (for all under /api/v1/*)
    Route::options('/{any}', fn () => response()->noContent())
        ->where('any', '.*');

    // Throttled group
    Route::middleware(['throttle:api'])->group(function () {
        // Health
        Route::get('/health', function () {
            return response()->json([
                'ok'      => true,
                'service' => config('app.name'),
                'env'     => config('app.env'),
                'version' => 'v1',
                'time'    => now()->toIso8601String(),
            ]);
        })->name('api.v1.health');

        // Holds API — register only if controller exists AND dependency is bound
        if (
            class_exists(\App\Http\Controllers\Holds\Api\V1\HoldsApiController::class)
            && app()->bound(\App\Domain\Holds\Services\Contracts\HoldGateway::class)
        ) {
            $holds = \App\Http\Controllers\Holds\Api\V1\HoldsApiController::class;

            Route::prefix('holds')->group(function () use ($holds) {
                Route::post('/', [$holds, 'store'])->name('api.v1.holds.store');

                Route::get('/{reference}', [$holds, 'show'])
                    ->where('reference', '[A-Za-z0-9\-_]+')
                    ->name('api.v1.holds.show');

                Route::get('/{reference}/status', [$holds, 'status'])
                    ->where('reference', '[A-Za-z0-9\-_]+')
                    ->name('api.v1.holds.status');

                Route::post('/{reference}/capture', [$holds, 'capture'])
                    ->where('reference', '[A-Za-z0-9\-_]+')
                    ->name('api.v1.holds.capture');

                Route::post('/{reference}/cancel', [$holds, 'cancel'])
                    ->where('reference', '[A-Za-z0-9\-_]+')
                    ->name('api.v1.holds.cancel');
            });
        }

        // Payments API (optional)
        if (class_exists(\App\Http\Controllers\Api\V1\PaymentsController::class)) {
            $payments = \App\Http\Controllers\Api\V1\PaymentsController::class;

            Route::prefix('payments')->group(function () use ($payments) {
                Route::post('/intents', [$payments, 'createIntent'])
                    ->name('api.v1.payments.intents.create');

                Route::post('/intents/{id}/confirm', [$payments, 'confirmIntent'])
                    ->where('id', '[A-Za-z0-9\-_]+')
                    ->name('api.v1.payments.intents.confirm');

                Route::post('/charges', [$payments, 'charge'])
                    ->name('api.v1.payments.charge');
            });
        }

        // API Keys verify (optional)
        if (class_exists(\App\Http\Controllers\Api\V1\ApiKeyController::class)) {
            $keys = \App\Http\Controllers\Api\V1\ApiKeyController::class;

            Route::get('/keys/verify', [$keys, 'verify'])
                ->name('api.v1.keys.verify');
        }

        // Authenticated (Sanctum/token guard)
        Route::middleware('auth:sanctum')->group(function () {
            Route::get('/me', function (Request $request) {
                return $request->user();
            })->name('api.v1.me');
        });
    });
});

/*
|--------------------------------------------------------------------------
| Fallback (JSON 404)
|--------------------------------------------------------------------------
*/
Route::fallback(function () {
    return response()->json([
        'ok'      => false,
        'error'   => 'Not Found',
        'message' => 'Route not found. Check path/method and API version.',
    ], 404);
});
