<?php

namespace Tests\Feature;

use App\Models\Deposit;
use App\Models\Job;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Mail\Mailable;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Schema;
use Tests\TestCase;

/**
 * Tiny Eloquent stub we can alias to missing app models in tests.
 */
class _EloquentStub extends Model {}

/**
 * Alias missing related models to a stub (no-ops) in the test runtime.
 * If your real app already defines these classes, class_exists prevents aliasing.
 */
if (! class_exists(\App\Models\Brand::class)) {
    class_alias(_EloquentStub::class, \App\Models\Brand::class);
}
if (! class_exists(\App\Models\Flow::class)) {
    class_alias(_EloquentStub::class, \App\Models\Flow::class);
}

/**
 * A minimal mailable we can assert against.
 */
class TestDepositReleasedMailable extends Mailable
{
    public function build(): self
    {
        return $this->subject('Deposit released')->html('<p>Deposit released</p>');
    }
}

/**
 * A minimal test runner for the auto-cancel “core”.
 * - No console IO or locks
 * - No Stripe calls (cancel always succeeds)
 * - Computes due time using config fallbacks
 */
class AutoCancelCoreRunner
{
    /** @var array<string,mixed> */
    protected array $opts = ['dry' => false, 'limit' => null];

    public function option($key = null)
    {
        if ($key === null) return $this->opts;
        return $this->opts[$key] ?? null;
    }

    protected function cancelStripeAuthorization(Deposit $deposit): bool
    {
        return true; // simulate success
    }

    protected function computeEffectiveDue(Deposit $deposit): ?Carbon
    {
        $defaultMinutes = (int) (config('holds.default_cancel_after_minutes', 720) ?: 720);
        $authWindowDays = (int) (config('holds.stripe_auth_window_days', 7) ?: 7);

        $planned = $deposit->planned_cancel_at
            ? Carbon::parse($deposit->planned_cancel_at)
            : (optional($deposit->job)->end_at
                ? Carbon::parse($deposit->job->end_at)->addMinutes($defaultMinutes)
                : null);

        $ceiling = $deposit->authorized_at
            ? Carbon::parse($deposit->authorized_at)->addDays($authWindowDays)
            : null;

        if ($planned && $ceiling) return $planned->min($ceiling);
        return $planned ?: $ceiling;
    }

    public function handle(): int
    {
        $now = Carbon::now();

        $q = Deposit::query()
            ->with('job')
            ->whereIn('status', ['authorized', 'authorised'])
            ->whereNotNull('stripe_payment_intent')
            ->orderBy('id');

        $processed = 0;

        foreach ($q->cursor() as $deposit) {
            if ($this->option('limit') !== null && $processed >= (int) $this->option('limit')) {
                break;
            }
            $processed++;

            $due = $this->computeEffectiveDue($deposit);
            if (! $due || $now->lt($due)) {
                continue; // not due yet
            }

            if ($this->option('dry')) {
                continue;
            }

            if (! $this->cancelStripeAuthorization($deposit)) {
                continue;
            }

            DB::transaction(function () use ($deposit) {
                $deposit->status = 'released'; // or 'canceled' if your UI expects that
                $deposit->released_at = Carbon::now();

                $meta = (array) ($deposit->meta ?? []);
                $meta['auto_cancel'] = [
                    'canceled_at' => Carbon::now()->toIso8601String(),
                    'reason'      => 'auto_cancel_if_no_capture',
                ];
                $deposit->meta = $meta;
                $deposit->save();

                DB::afterCommit(function () use ($deposit) {
                    $job = $deposit->job;
                    if ($job && !empty($job->customer_email)) {
                        Mail::to($job->customer_email)->send(new TestDepositReleasedMailable());
                    }
                });
            });
        }

        return 0;
    }
}

final class AutoCancelHoldsTest extends TestCase
{
    protected function setUp(): void
    {
        parent::setUp();

        // ---- Minimal sqlite schemas (simple; no hasColumn calls) ----

        if (! Schema::hasTable('brands')) {
            Schema::create('brands', function (Blueprint $t) {
                $t->bigIncrements('id');
                $t->string('name')->nullable();
                $t->timestamps();
                $t->softDeletes();
            });
        }

        if (! Schema::hasTable('flows')) {
            Schema::create('flows', function (Blueprint $t) {
                $t->bigIncrements('id');
                $t->timestamps();
                $t->softDeletes();
            });
        }

        if (! Schema::hasTable('jobs')) {
            Schema::create('jobs', function (Blueprint $t) {
                $t->bigIncrements('id');
                $t->unsignedBigInteger('brand_id')->nullable()->index();
                $t->unsignedBigInteger('flow_id')->nullable()->index();
                $t->dateTime('end_at')->nullable();
                $t->string('customer_email')->nullable();
                $t->timestamps();
                $t->softDeletes();
            });
        }

        if (! Schema::hasTable('deposits')) {
            Schema::create('deposits', function (Blueprint $t) {
                $t->bigIncrements('id');

                $t->unsignedBigInteger('job_id')->nullable()->index();
                $t->unsignedBigInteger('customer_id')->nullable();

                $t->integer('authorized_cents')->default(0);
                $t->integer('captured_cents')->default(0);
                $t->string('currency', 3)->default('NZD');

                $t->string('reference')->nullable();
                $t->string('stripe_payment_intent')->nullable();
                $t->string('stripe_charge')->nullable();
                $t->string('stripe_payment_method')->nullable();

                $t->string('status')->default('authorized');
                $t->string('last_reason')->nullable();
                $t->json('meta')->nullable();

                $t->dateTime('authorized_at')->nullable();
                $t->dateTime('planned_cancel_at')->nullable();
                $t->dateTime('released_at')->nullable();

                $t->timestamps();
                $t->softDeletes();
            });
        }
    }

    public function test_releases_due_holds_and_sends_email(): void
    {
        Carbon::setTestNow(Carbon::parse('2025-09-14 10:00:00'));

        $job = Job::query()->forceCreate([
            'brand_id'       => null,
            'flow_id'        => null,
            'end_at'         => now()->subHours(2),
            'customer_email' => 'test@example.com',
        ]);

        $deposit = Deposit::query()->forceCreate([
            'job_id'                => $job->id,
            'status'                => 'authorized',
            'currency'              => 'NZD',
            'authorized_cents'      => 0,
            'captured_cents'        => 0,
            'stripe_payment_intent' => 'pi_test',
            'authorized_at'         => now()->subHour(),
            'planned_cancel_at'     => now()->subMinutes(5),
            'meta'                  => [],
        ]);

        Mail::fake();

        // Run the core runner (no IO / no Stripe / deterministic)
        (new AutoCancelCoreRunner())->handle();

        // Assert DB updated
        $deposit->refresh();
        $this->assertTrue(in_array($deposit->status, ['released', 'canceled'], true), 'Deposit should be released/canceled.');
        $this->assertNotNull($deposit->released_at, 'released_at should be set.');

        // Assert our named mailable was sent exactly once
        Mail::assertSent(TestDepositReleasedMailable::class, 1);

        Carbon::setTestNow();
    }
}
